/*****************************************************************************
* File Name          : ISO15693_Category.c
* Author             : CrystalSu
* Version            : V1.0.0.0
* Date               : Feb 14, 2020
* Description        : 
*                         - ISO14443B_Transparent_Configure_CRC
*                         - ISO15693_Transparent_With_RxMultiple
******************************************************************************/   

#include "BC45_Board_Define.h"
#include "Common_Utility_Category.h"
#include "ISO15693_Category.h"
#include "BC45_Chip_Function.h"
#include "Delay.h"

// Global variable 
tISO15693_Property  ISO15693_Property ;

/*******************************************************************************
* Function Name : ISO15693_Command
* Description   : Perform 
*				  	- air interface commands following ISO15693
*					- setting up ISO15693 configuration
*					- setting and getting RF communication speed
*			
* Input         : 	- Command : Commands associated to ISO15693 as shown in source below 
*					- Param	: Parameters associated to command as shown in source below (after CMD_TYPE)
*					- LenParam : Length of input parameters
* Output        : 
*					- Data_Resp	: Reponse from operation the command
*					- LenData_Resp : Length of output response
* Return        : 
*				   - Response : Result from operating the function
*				 	 	- _SUCCESS_ 
*				 	 	- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Command( uint8_t Command, uint8_t *Param, uint16_t LenParam, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t   Resp ;
	uint8_t   Flg ;
	uint8_t   temp_UID[8] ;
	uint8_t   Specific_Param[64] ;
	uint16_t  Len_Specific_Param ;
	uint8_t   Mask_Value[8] ;
	uint8_t   Inventory_Mode, AFI, Mask_Len ;
	uint8_t   Non_Inventory_Mode ;
	uint8_t   Num_Block ;
	uint8_t   Pwd_Mode, PA, Len_Pwd_Bytes ;
	volatile  uint8_t  Block_Size ;
	
	switch(Command)
	{
		case D_Config_ISO15693 :// Config At Reader
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + LRC									
								Resp = ISO15693_Config( 0x00 ) ;
								ISO15693_Property.Speed = 0x00 ;
								*(LenData_Resp) = 0 ; 
		break ;

		case D_Config_Speed_Reader : // Config At Reader
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Speed_Sel + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + LRC									
								if ( LenParam >= 1 )
								{ 
									ISO15693_Property.Speed = *( Param ) ;
									BC45_Speed_15693( ISO15693_Property.Speed ) ;
									Resp = _SUCCESS_ ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
								}
								*(LenData_Resp) = 0 ; 
		break ;

		case D_Get_Speed_Reader : // Speed at Reader
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + Speed + LRC									
								Resp = ISO15693_Get_Speed_Reader( Data_Resp, LenData_Resp ) ;
		break ;

		case D_Inventory_1_Slot :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Inventory_Mode + AFI(Optional) + Mask_Len + Mask_Value(0) + ... + Mask_Value(N-1) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Inventory_Mode( Param, LenParam, &Inventory_Mode, &AFI, &Mask_Len, Mask_Value ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{	
									// No Specific_Param
									Resp = ISO15693_Inv_Req_1_Slot( ISO15693_Property.Speed, Inventory_Mode, AFI, Mask_Len, Mask_Value, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Inventory_16_Slots :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Inventory_Mode + AFI(Optional) + Mask_Len + Mask_Value(0) + ... + Mask_Value(N-1) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Inventory_Mode( Param, LenParam, &Inventory_Mode, &AFI, &Mask_Len, Mask_Value ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{	
									// No Specific_Param
									Resp = ISO15693_Inv_Req_16_Slots( ISO15693_Property.Speed, Inventory_Mode, AFI, Mask_Len, Mask_Value, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Stay_Quiet :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{	
									// No Specific_Param
									Resp = ISO15693_Stay_Quiet( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Select :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{	
									// No Specific_Param
									Resp = ISO15693_Select( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}

		break ;

		case D_Reset_to_Ready :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{	
									// No Specific_Param
									Resp = ISO15693_Reset_to_Ready( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Read_Single_Block :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + Block_Num + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 ) // Block_Num 1 Byte
									{	
										// Specific_Param = Block_Num( 1 Byte ) = 1 Byte
										Resp = ISO15693_Read_Single_Block( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, *(Specific_Param), Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Write_Single_Block :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + Block_Num + DataWr[0] + ... + DataWr[N-1] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 2 ) // Block_Number + Data_Wr ( At least 1 Byte ) = 2 Bytes
									{
										// Specific_Param = Block_Number + Data_Wr ( At least 1 Byte )
										Block_Size = ( uint8_t )( Len_Specific_Param - 1 )  ; 
										Resp = ISO15693_Write_Single_Block( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Block_Size, Specific_Param, Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;
		
		case D_Lock_Block :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + Block_Num + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 ) // Block Number 1 Byte
									{
										// Specific_Param = Block_Num( 1 Byte ) = 1 Byte
										Resp = ISO15693_Lock_Block( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, *(Specific_Param), Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Read_Multiple_Blocks :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + First_Block_Num + Num_Block + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 2 )  // Block_Num + Num_Block
									{	
										// Specific_Param = First_Block_Num( 1 Byte ) + Num_Block( 1 Byte ) = 2 Bytes
										Resp = ISO15693_Read_Multiple_Blocks( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Specific_Param, Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Write_Multiple_Blocks :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + Block_Size + First_Block_Num + Num_Block + DataWr[0] + ... + DataWr[N-1] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 3 ) // Block_Number + Block Size
									{
										Block_Size = Specific_Param[0] ;
										Num_Block = Specific_Param[2] ;
										if ( Len_Specific_Param >= (Block_Size * (Num_Block + 1)) + 3 ) // Block_Number + Num_Block + Block_Size + DataWr ( Block_Size*(Num_Block + 1) )
										{
											Resp = ISO15693_Write_Multiple_Blocks( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Block_Size, (Specific_Param + 1), Data_Resp, LenData_Resp ) ;
										}
										else
										{
											Resp = PARAMETER_NOT_CORRECT ;
											*(LenData_Resp) = 0 ; 
										}
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Write_AFI :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + AFI_Value + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 ) // AFI_Value 1 Byte
									{	
										// Specific_Param = AFI_Value( 1 Byte )
										Resp = ISO15693_Write_AFI( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, *( Specific_Param ), Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Lock_AFI :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + AFI_Value + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{	
									// No Specific_Param
									Resp = ISO15693_Lock_AFI( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;
		
		case D_Write_DSFID :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + DSFID_Value + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 ) // DSFID_Value 1 Byte
									{
										// Specific_Param = DSFID_Value( 1 Byte )
										Resp = ISO15693_Write_DSFID( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, *( Specific_Param ), Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Lock_DSFID :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + AFI_Value + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Lock_DSFID( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Get_System_Information :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Get_System_Information( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Get_multiple_block_security_status :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + First_Block_Num + Num_Block + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 2 )  // First_Block_Num + Num_Block
									{	
										// Specific_Param = First_Block_Num( 1 Byte ) + Num_Block( 1 Byte ) = 2 Bytes
										Resp = ISO15693_Get_Multiple_Block_Security_Status( ISO15693_Property.Speed, Non_Inventory_Mode, temp_UID, Specific_Param, Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

// ---------------------------------------------------------- HTK CUSTOM COMMAND -------------------------------------------------------------------------------------------------------------
	
		case D_Set_EAS_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{	
									// No Specific_Param
									Resp = ISO15693_Set_EAS_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_PHILIP, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Reset_EAS_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Reset_EAS_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_PHILIP, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Lock_EAS_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Lock_EAS_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_PHILIP, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_EAS_Alarm_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] ( UID is Optional ) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_EAS_Alarm_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_PHILIP, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Kill_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + Kill_Code[0] + ... + Kill_Code[3] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 4 )  // Kill_Code( 4 Bytes )
									{
										// Specific_Param = Kill Code( 4 Bytes ) = 4 Bytes
										Resp = ISO15693_Kill_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Specific_Param, Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}

		break ;


		case D_Set_Password_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + Pwd_Mode + RNG[0] + ... + RNG[N-1] + Pwd[0] + ... + Pwd[N-1] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 ) // Pwd_Mode
									{
										Pwd_Mode = Specific_Param[0] ;
										PA =  ( Pwd_Mode & PA_MASK ) >> 2 ;
										if ( PA == PM_USE_PWD_64_PROTECT_ALL ) 
										{
											Len_Pwd_Bytes = 8 ; // Pwd_Mode + RNG ( 8 Bytes ) + Pwd ( 8 Bytes )
										}
										else
										{
											Len_Pwd_Bytes = 4 ; // Pwd_Mode + RNG ( 4 Bytes ) + Pwd ( 4 Bytes )
										}
										
										if ( Len_Specific_Param >= ( Len_Pwd_Bytes * 2 ) + 1 ) 
										{
											Resp = ISO15693_Set_Password_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Pwd_Mode, (Specific_Param + 1), (Specific_Param + Len_Pwd_Bytes + 1), Data_Resp, LenData_Resp ) ;
										}
										else
										{
											Resp = PARAMETER_NOT_CORRECT ;
											*(LenData_Resp) = 0 ; 
										}
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Write_Password_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + Pwd_ID + Pwd[0] + ... Pwd[3] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 5 )  // Pwd_ID + Pwd ( 4 Bytes ) = 5 Bytes
									{	
										// Specific_Param = Pwd_ID( 1 Byte ) + Pwd( 4 Bytes ) = 5 Bytes
										Resp = ISO15693_Write_Password_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Specific_Param, Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Lock_Password_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + Pwd_ID + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 )  // Pwd_ID
									{
										// Specific_Param = Pwd_ID( 1 Byte ) = 1 Byte
										Resp = ISO15693_Lock_Password_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, *( Specific_Param ), Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;
	
		case D_Set_Password_Mode_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + Pwd_Mode + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 )  // Pwd_Mode
									{
										// Specific_Param = Pwd_Mode( 1 Byte ) = 1 Byte
										Resp = ISO15693_Set_Password_Mode_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, *( Specific_Param ), Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Lock_Password_Mode_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Lock_Password_Mode_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Get_Password_Mode_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Get_Password_Mode_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;
		
		case D_Set_OTP_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + OTP_Mode + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 1 )  // OTP_Mode
									{
										// Specific_Param = OTP_Mode( 1 Byte ) = 1 Byte
										Resp = ISO15693_Set_OTP_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, *( Specific_Param ), Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Get_OTP_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Get_OTP_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Write_OTP_UID_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + OTP_UID[0] + ... OTP_UID[5] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									if ( Len_Specific_Param >= 6 )  // OTP_UID ( 6 Bytes )
									{
										// Specific_Param = OTP_UID( 6 Bytes ) = 6 Bytes
										Resp = ISO15693_Write_OTP_UID_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Specific_Param, Data_Resp, LenData_Resp ) ;
									}
									else
									{
										Resp = PARAMETER_NOT_CORRECT ;
										*(LenData_Resp) = 0 ; 
									}
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Read_OTP_UID_HTK :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + Non_Inventory_Mode + UID[0] + ... UID[7] + OTP_UID[0] + ... OTP_UID[5] ( UID is Optional ) + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								Flg = ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( Param, LenParam, &Non_Inventory_Mode, temp_UID, Specific_Param, &Len_Specific_Param  ) ;
								if ( Flg != PARAMETER_NOT_CORRECT )
								{
									// No Specific_Param
									Resp = ISO15693_Read_OTP_UID_HTK( ISO15693_Property.Speed, Non_Inventory_Mode, IC_MFG_HTK, temp_UID, Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;
		

		case D_Transparent_With_CRC :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD  + TimeOut + Data_Tx[0] + ... + Data_Tx[n + 1] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								if ( LenParam >= 2 ) 
								{ 
									Resp = Transparent_With_CRC( (Param + 1), (LenParam - 1), *(Param), Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Transparent_Without_CRC :
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + TimeOut + Data_Tx[0] + ... + Data_Tx[n + 1] + LRC
								// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
								if ( LenParam >= 2 ) 
								{ 
									Resp = Transparent_Without_CRC( (Param + 1), (LenParam - 1), *(Param), Data_Resp, LenData_Resp ) ;
								}
								else
								{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
								}
		break ;

		case D_Send_1_Pulse :
					// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + LRC
					// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
					Resp = ISO15693_Send_1_Pulse( Data_Resp, LenData_Resp ) ;
					break ;
                
		case  D_Transparent_Config_CRC :                
                                                                // SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + TimeOut + TxCRCEn + RxCRCEn + Data_Tx[0] + ... + Data_Tx[n + 1] + LRC
					// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RESP + DATA[0] + ... + DATA[N-1] + LRC									
					if ( LenParam >= 4 ) 
					{ 
						Resp = Transparent_Configure_CRC( *(Param + 1), *(Param + 2), (Param + 3), (LenParam - 3), *(Param), Data_Resp, LenData_Resp ) ;
					}
					else
					{
						Resp = PARAMETER_NOT_CORRECT ;
						*(LenData_Resp) = 0 ; 
					}
                
					break;  
		
		case  D_Transparent_With_RxMultiple :                
					// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + CMD + TimeOut + Data_Tx[0] + ... + Data_Tx[n + 1] + LRC
					// SOP + LEN_H + LEN_L + SEQ_NUM + DEV_ID + CMD_TYPE + FBP_CMD + RxSolt[0] + Resp[0] + Len[0] + Data[0][0] + ... + Data[0][N-1] + ... + RxSlot[m] + Resp[m] + Len[m] + Data[m][0] + ... + Data[m][N-1] +  LRC									
                 
					if ( LenParam >= 2 ) 
					{ 
								Resp = Transparent_With_RxMultiple( (Param + 1),  (LenParam - 1), *(Param), Data_Resp, LenData_Resp ) ;
					}
					else
					{
									Resp = PARAMETER_NOT_CORRECT ;
									*(LenData_Resp) = 0 ; 
					}
					break;  

		default :
					Resp = UNKNOWN_CMD ;
					*(LenData_Resp) = 0 ; 
		break ;
	}

	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Check_And_Separate_Param_Inventory_Mode
* Description 	: Check Length of received parameters package for Inventory command from host and 
*				  extracts to Inventory_Mode, AFI, Mask_Len, Mask_Value
* Input         : 
*				  - Param_Package : Received Parameters Package for Inventory command 
*				  - LenParam_Package : Length of Received Parameters Package for Inventory command
* Output		: 
*				  - Inventory_Mode : RFU_Flg(1 bit) + Option Flg(1 bit) + Protocol Extension Flg(1 bit) + '0' + AFI mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : AFI mode (Bit3:0)
*							- 0000b - Non Check AFI
*							- 0001b - Check AFI  
*				  - AFI : Application family identifier 
*					
*				  - Mask_Len : The mask length indicates the number of significant bits to transmit.
*				  - Mask_Value : The mask value is contained in an integer number of bytes
*					Note : If the mask length is not a multiple of 8 (bits), 
*						   the mask value MSB shall be padded with the required number of null (set to 0) bits 
*						   so that the mask value is contained in an integer number of bytes.
* Return        : 
*				  - Status
*						- _SUCCESS_  
*						- PARAMETER_NOT_CORRECT
*******************************************************************************/
uint8_t ISO15693_Check_And_Separate_Param_Inventory_Mode( uint8_t *Param_Package, uint16_t LenParam_Package, uint8_t *Inventory_Mode, uint8_t *AFI , uint8_t *Mask_Len, uint8_t *Mask_Value ) 
{
	uint8_t  i ;
	uint8_t  Flg ;
	uint8_t  temp_Inventory_Mode ;
	uint8_t  temp_Mask_Len, Mask_Value_Pos, Mask_Byte ;
	uint8_t  LenParam_Complete ;
	
	Flg = 0 ;

	if ( LenParam_Package >= 1 ) // Inventory_Mode
	{ 
		temp_Inventory_Mode = *( Param_Package ) ;
		*( Inventory_Mode ) = temp_Inventory_Mode ;
		if ( ( temp_Inventory_Mode & AFI_MODE_MASK ) == NON_CHECK_AFI_MODE )
		{
			if ( LenParam_Package >= 2 ) //  Inventory_Mode + Mask_Len
			{
				*( AFI ) = 0x00 ;
				temp_Mask_Len = *( Param_Package + 1 ) ;									
				Mask_Value_Pos = 2 ;
				LenParam_Complete = 2 ;
			}
			else
			{
				Flg = PARAMETER_NOT_CORRECT ;
			}
		}
		else
		{
			if ( LenParam_Package >= 3 ) //  Inventory_Mode + AFI + Mask_Len
			{
				*( AFI ) = *( Param_Package + 1 ) ;
				temp_Mask_Len = *( Param_Package + 2 ) ;									
				Mask_Value_Pos = 3 ;
				LenParam_Complete = 3 ;
			}
			else
			{
				Flg = PARAMETER_NOT_CORRECT ;
			}
		}

		if ( Flg != PARAMETER_NOT_CORRECT )
		{
			Mask_Byte = temp_Mask_Len / 8 ;
			if ( (temp_Mask_Len % 8) != 0 )
			{	
				Mask_Byte = Mask_Byte + 1 ;	
			}
			LenParam_Complete += Mask_Byte ; 
			if ( LenParam_Package >= LenParam_Complete )
			{
				*( Mask_Len ) = temp_Mask_Len ;
				for ( i = 0 ; i < Mask_Byte ; i++ )
				{
					*( Mask_Value + i ) = *( Param_Package + Mask_Value_Pos + i ) ; 	
				}
			}
			else
			{
				Flg = PARAMETER_NOT_CORRECT ;
			}
		}
		else
		{
			Flg = PARAMETER_NOT_CORRECT ;
		}
	}
	else
	{
		Flg = PARAMETER_NOT_CORRECT ;
	}

	return Flg ;	
}

/*******************************************************************************
* Function Name	: ISO15693_Check_And_Separate_Param_Non_Inventory_Mode
* Description 	: Check Length of received parameters package for Non Inventory command and
*				  extracted to Non_Inventory_Mode, UID, Other Parameter
* Input         : 
*				  - Param_Package : Received Parameters Package for Non Inventory command 
*				  - LenParam_Package : Length of Received Parameters Package for Non Inventory command
* Output		: 
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode(Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID_Frm_Param : UID from parameter package
*				  - Other_Param : Specific parameters for each command in ISO15693
*				  - Len_Other_Param : Length of Specific parameters for each command in ISO15693
* Return        : 
*				  - Status
*						- _SUCCESS_  
*						- PARAMETER_NOT_CORRECT
*******************************************************************************/
uint8_t ISO15693_Check_And_Separate_Param_Non_Inventory_Mode( uint8_t *Param_Package, uint16_t LenParam_Package, uint8_t *Non_Inventory_Mode, uint8_t *UID_Frm_Param , uint8_t *Other_Param, uint16_t *Len_Other_Param ) 
{
	uint8_t  i ; 
	uint8_t  Flg ; 

	Flg = _SUCCESS_ ;

	if ( LenParam_Package >= 9 )  // Non_Inventory_Mode + UID ( 8 Bytes )
	{
		*(Non_Inventory_Mode) = *( Param_Package ) ;
		// Get UID from package
		for ( i = 0 ; i < ISO15693_LEN_UID_BYTE ; i++ )
		{
			*( UID_Frm_Param + i ) = *( Param_Package + i + 1 ) ;	
		}

		*(Len_Other_Param) = ( LenParam_Package - ISO15693_LEN_UID_BYTE - 1 ) ;
		for ( i = 0 ; i < *(Len_Other_Param) ; i++ )
		{
			*( Other_Param + i ) = *( Param_Package + ISO15693_LEN_UID_BYTE + 1 + i ) ;	
		}
	}
	else
	{
		Flg = PARAMETER_NOT_CORRECT ;
	}
	
	return Flg ;	
}

/*******************************************************************************
* Function Name	: ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional
* Description 	: Check Length of received parameters package for Non Inventory command (UID is optional) and 
*				  extracted to Non_Inventory_Mode, UID, Other Parameter from it 
* Input         : 
*				  - Param_Package : Received Parameters Package for Non Inventory command 
*				  - LenParam_Package : Length of Received Parameters Package for Non Inventory command
* Output		: 
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode(Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID_Frm_Param : UID from parameter package
*				  - Other_Param : Specific parameters for each command in ISO15693
*				  - Len_Other_Param : Length of Specific parameters for each command in ISO15693
* Return        : 
*				  - Status
*						- _SUCCESS_  
*						- PARAMETER_NOT_CORRECT
*******************************************************************************/
uint8_t ISO15693_Check_And_Separate_Param_Non_Inventory_Mode_UID_Is_Optional( uint8_t *Param_Package, uint16_t LenParam_Package, uint8_t *Non_Inventory_Mode, uint8_t *UID_Frm_Param , uint8_t *Other_Param, uint16_t *Len_Other_Param ) 
{
	uint8_t  i ; 
	uint8_t  Flg ; 
	uint8_t  temp_Non_Inventory_Mode, Operation_Mode ; 

	Flg = _SUCCESS_ ;

	if ( LenParam_Package >= 1 )
	{
		temp_Non_Inventory_Mode = *( Param_Package ) ;
		Operation_Mode = temp_Non_Inventory_Mode & OPERATION_MODE_MASK ;
		
		if ( Operation_Mode == ADDR_MODE )
		{
			if ( LenParam_Package >=  ISO15693_LEN_UID_BYTE + 1 )
			{	
				*(Non_Inventory_Mode) = temp_Non_Inventory_Mode ;
				// Get UID from package
				for ( i = 0 ; i < ISO15693_LEN_UID_BYTE ; i++ )
				{
					*( UID_Frm_Param + i ) = *( Param_Package + i + 1 ) ;	
				}

				*(Len_Other_Param) = ( LenParam_Package - ISO15693_LEN_UID_BYTE - 1 ) ;
				for ( i = 0 ; i < *(Len_Other_Param) ; i++ )
				{
					*( Other_Param + i ) = *( Param_Package + ISO15693_LEN_UID_BYTE + 1 + i ) ;	
				}
			}
			else
			{
				Flg = PARAMETER_NOT_CORRECT ;
			}
		}
		else
		{
			*(Non_Inventory_Mode) = temp_Non_Inventory_Mode ;
			// Force UID = 0x00
			for ( i = 0 ; i < ISO15693_LEN_UID_BYTE ; i++ )
			{
				*( UID_Frm_Param + i ) = 0x00 ;	
			}

			*(Len_Other_Param) = ( LenParam_Package - 1 ) ;
			for ( i = 0 ; i < *(Len_Other_Param) ; i++ )
			{
				*( Other_Param + i ) = *( Param_Package + 1 + i ) ;	
			}
		}
	}
	else
	{
		Flg = PARAMETER_NOT_CORRECT ;
	}
	
	return Flg ;
}

/*******************************************************************************
* Function Name	: ISO15693_Check_Response
* Description 	: Check Execution response and Process received raw data from air
* Input         : 
*				  - Resp_RxRF : Execution Response related to BC45 from function BC45_CheckErr
*				  - Data_RxRF : Recieved raw data from executing the RF-related function. i.e. ISO15693_Read_Single_Block.  
*				  - LenData_RxRF : Length of Recieved raw data
*
* Output		: 
*				  - Data_Resp  	 : processed data which can be data or error message from tag
*				  - LenData_Resp : Length of processed data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Check_Response( uint8_t Resp_RxRF, uint8_t *Data_RxRF, uint16_t LenData_RxRF, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint16_t  i ;
	uint8_t   Resp ;
	uint8_t   Response_Flg ;
	
	Resp = Resp_RxRF ;
	if ( Resp_RxRF == _SUCCESS_ )
	{
		if ( LenData_RxRF > 0 )
		{
			Response_Flg = *( Data_RxRF ) ;
			if ( (Response_Flg & ERR_FLAG_MASK) == ERR_FLAG_MASK )
			{
				Resp = FLAG_ERR ; 
			}

			*( LenData_Resp ) = LenData_RxRF - 1 ;
			for ( i = 0; i < LenData_RxRF - 1 ; i++ )
			{
				*( Data_Resp + i ) = Data_RxRF[i + 1] ;
			}
		}
		else // Success But No Data
		{
			Resp = INVALID_RESP ;  
			*( LenData_Resp ) = 0 ;
		}
	}
	else // Err 
	{
		*( LenData_Resp ) = LenData_RxRF ;
		for ( i = 0; i < LenData_RxRF ; i++ )
		{
			*( Data_Resp + i ) = Data_RxRF[i] ;
		}
	}
	
	return Resp ;
}

/*******************************************************************************
* Function Name : ISO15693_Config
* Description   : Configure Registers and RF-Communication speed for ISO15693 Operation
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits)	
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
* Output        : None
* Return        : 
*				  - Response
*				 		- _SUCCESS_ 
*******************************************************************************/
uint8_t ISO15693_Config( uint8_t Speed ) 
{
	BC45_Config_15693() ;
	BC45_Speed_15693( Speed ) ;
	
	// ON Field
	BC45_ON_RF( BC45_BOARD_PROPERTY.Driver_Config_Type ) ;
	Delay_ms( 10 ) ;
	
	return _SUCCESS_ ;
} 

/*******************************************************************************
* Function Name	: ISO15693_Get_Speed_Reader
* Description 	: Return current RF-Speed setting
* Input         : None
* Output        : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits)	
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				: - LenSpeed : Length of output, always = 1
* Return        : 
*				  - Response
*						- _SUCCESS_ 
*******************************************************************************/
uint8_t ISO15693_Get_Speed_Reader( uint8_t *Speed, uint16_t *LenSpeed )
{	 
	*( Speed )    = ISO15693_Property.Speed ;
	*( LenSpeed ) = 1 ;

	return _SUCCESS_ ;	
}

/*******************************************************************************
* Function Name : Request_Flag_ISO15693
* Description   : Generate flags in resquest format of ISO15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits)	
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Mode_Value : Inventory_Mode, Non_Inventory_Mode from parameter packet
*				  - Mode 
*						- INVENTORY_1_SLOT_MODE(0x00)
*						- INVENTORY_16_SLOTS_MODE(0x01)
*						- NON_INVENTORY_MODE(0x02)
* Output        : None
* Return        : 
*				   - Request_Flg : Request flag of ISO15693
*******************************************************************************/
uint8_t Request_Flag_ISO15693( uint8_t Speed, uint8_t Mode_Value, uint8_t Mode )
{
	uint8_t  Request_Flg ;
	uint8_t  Speed_Rx ;
	uint8_t  Operation_Mode ;
	uint8_t  AFI_Mode ;
	
	Request_Flg = 0x00 ;
	
	Speed_Rx = ( Speed & 0x0F ) ;
	if ( Speed_Rx == SPEED_1_SUB_LOW ) 	 // 1 sub low
	{	
		;	
	}
	else if	( Speed_Rx == SPEED_1_SUB_ULTRA_HIGH ) // 1 Ultra high	
	{	
		Request_Flg |= DATA_RATE_FLAG_SET_MASK ;	
	}
	else if	( Speed_Rx == SPEED_2_SUB_LOW ) // 2 sub low	
	{	
		Request_Flg |= SUB_CARRIER_FLAG_SET_MASK ;	
	}
	else if	( Speed_Rx == SPEED_2_SUB_HIGH ) // 2 sub high		
	{	
		Request_Flg |= ( DATA_RATE_FLAG_SET_MASK | SUB_CARRIER_FLAG_SET_MASK ) ;	
	}
	else 						// 1 sub high
	{	
		Request_Flg |= DATA_RATE_FLAG_SET_MASK ;	
	};

	if ( Mode == INVENTORY_1_SLOT_MODE ) 
	{
		Request_Flg |= ( INVENTORY_FLAG_SET_MASK | NB_SLOTS_FLAG_SET_MASK )  ;	

		AFI_Mode = ( Mode_Value & AFI_MODE_MASK ) ;
		if ( AFI_Mode == CHECK_AFI_MODE )
		{
			Request_Flg |= AFI_FLAG_SET_MASK  ;	
		}
	}
	else if ( Mode == INVENTORY_16_SLOTS_MODE ) 
	{
		Request_Flg |= INVENTORY_FLAG_SET_MASK  ;	

		AFI_Mode = ( Mode_Value & AFI_MODE_MASK ) ;
		if ( AFI_Mode == CHECK_AFI_MODE )
		{
			Request_Flg |= AFI_FLAG_SET_MASK  ;	
		}
	}
	else // Non-Inventory mode
	{
		Operation_Mode = ( Mode_Value & OPERATION_MODE_MASK ) ;
		if ( Operation_Mode == ADDR_MODE )
		{
			Request_Flg |= ADDR_FLAG_SET_MASK  ;	
		}
		else if ( Operation_Mode == SEL_MODE )
		{
			Request_Flg |= SEL_FLAG_SET_MASK  ;	
		}
	};

	if ( (Mode_Value & RFU_FLAG_IN_MODE_MASK) == RFU_FLAG_IN_MODE_MASK )
	{	
		Request_Flg |= RFU_FLAG_SET_MASK  ;	
	}

	if ( (Mode_Value & OPTION_FLAG_IN_MODE_MASK) == OPTION_FLAG_IN_MODE_MASK )
	{	
		Request_Flg |= OPTION_FLAG_SET_MASK  ;	
	}

	if ( (Mode_Value & PROTOCOL_EX_FLAG_IN_MODE_MASK) == PROTOCOL_EX_FLAG_IN_MODE_MASK )
	{	
		Request_Flg |= PROTOCOL_EX_FLAG_SET_MASK  ;	
	}

	return Request_Flg ;
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//



//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//
//--------------------------ISO15693 Standard Cmd --------------------------------------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//

/*******************************************************************************
* Function Name	: ISO15693_Inv_Req_1_Slot
* Description 	: Perform Inventory command in ISO15693 standard
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Inv_Mode : RFU_Flg(1 bit) + Option Flg(1 bit) + Protocol Extension Flg(1 bit) + '0' + AFI mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : AFI mode (Bit3:0)
*							- 0000b : Non Check AFI
*							- 0001b : Check AFI  
*				  - AFI : Application family identifier 
*				  - Mask_Len : The mask length indicates the number of significant bits.
*				  - Mask_Value : The mask value is contained in an integer number of bytes
*					Note : If the mask length is not a multiple of 8 (bits), 
*						   the mask value MSB shall be padded with the required number of null (set to 0) bits 
*						   so that the mask value is contained in an integer number of bytes.
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Inventory command
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Inv_Req_1_Slot( uint8_t Speed, uint8_t Inv_Mode, uint8_t AFI, uint8_t Mask_Len, uint8_t *Mask_Value, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint16_t  i ;
	uint8_t   Resp ;
	uint8_t   Mask_Byte, Mask_Value_Pos ;

	Mask_Byte = 0;

	// Request Flag
	Data_TxRF[0] = Request_Flag_ISO15693( Speed, Inv_Mode, INVENTORY_1_SLOT_MODE ) ;

	Data_TxRF[1] = ISO15693_INVENTORY_RF_CMD ;		//Inventory Command
	if ( ( Inv_Mode & AFI_MODE_MASK ) == NON_CHECK_AFI_MODE )
	{
		Data_TxRF[2] = Mask_Len ;	//Mask Length
	
		Mask_Value_Pos = 3 ;
		LenData_TxRF =  3 ;
	}
	else
	{
		Data_TxRF[2] = AFI ; 		//AFI
		Data_TxRF[3] = Mask_Len ;	//Mask Length
	
		Mask_Value_Pos = 4 ;
		LenData_TxRF = 4 ;
	}

	Mask_Byte = Mask_Len / 8 ;
	if ( (Mask_Len % 8) != 0 )
	{	
		Mask_Byte = Mask_Byte + 1 ;	
	}

	for( i = 0 ; i < Mask_Byte ; i++ ) 
	{	
		Data_TxRF[Mask_Value_Pos + i] = *( Mask_Value + i ) ;		//Mask Length	
	}

	LenData_TxRF += Mask_Byte ;
	
	BC45_CRC_Setting( TxCRC_Enable, RxCRC_Enable ) ;

	Resp = BC45_Transceive( &Data_TxRF[0], LenData_TxRF, &Data_RxRF[0], &LenData_RxRF ) ;	

	if ( Resp == _SUCCESS_ )
	{
		if ( LenData_RxRF == LEN_ISO15693_INVENTORY_RF_RESP )
		{
			*(LenData_Resp) = LenData_RxRF - 1 ;	
			for( i = 0 ; i < LenData_RxRF - 1 ; i++ )
			{
				*( Data_Resp + i ) = Data_RxRF[i + 1] ;		
			}
		}
		else
		{
			Resp = INVALID_RESP ;
			*(LenData_Resp) = LenData_RxRF ;	
			for( i = 0 ; i < LenData_RxRF ; i++ )
			{
				*( Data_Resp + i ) = Data_RxRF[i] ;		
			}
		}
	}
	else
	{
		*(LenData_Resp) = LenData_RxRF ;	
		for( i = 0 ; i < LenData_RxRF ; i++ )
		{
			*( Data_Resp + i ) = Data_RxRF[i] ;		
		}
	}	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Inv_Req_16_Slots
* Description 	: Perform Inventory command in ISO15693 standard
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Inv_Mode : RFU_Flg(1 bit) + Option Flg(1 bit) + Protocol Extension Flg(1 bit) + '0' + AFI mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : AFI mode (Bit3:0)
*							- 0000b : Non Check AFI
*							- 0001b : Check AFI  
*				  - AFI : Application family identifier 
*				  - Mask_Len : The mask length indicates the number of significant bits.
*				  - Mask_Value : The mask value is contained in an integer number of bytes
*					Note : If the mask length is not a multiple of 8 (bits), 
*						   the mask value MSB shall be padded with the required number of null (set to 0) bits 
*						   so that the mask value is contained in an integer number of bytes.
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Inventory command
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Inv_Req_16_Slots( uint8_t Speed, uint8_t Inv_Mode, uint8_t AFI, uint8_t Mask_Len, uint8_t *Mask_Value, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint16_t  i ;
	uint8_t   Resp ;
	uint8_t   AFI_Mode ;
	uint8_t   Speed_Rx ;
	uint8_t   SlotMarker ;
	uint8_t   Mask_Byte, Mask_Value_Pos ;
	uint16_t  temp_LenData_Resp ;

	BC45_CRC_Setting( TxCRC_Enable, RxCRC_Enable ) ;

	Mask_Byte = 0 ;
	temp_LenData_Resp = 0 ;
	
	Speed_Rx = Speed & 0x0F ;

	AFI_Mode = Inv_Mode & AFI_MODE_MASK ;	
	
	// Request Flag
	Data_TxRF[0] = Request_Flag_ISO15693( Speed, Inv_Mode, INVENTORY_16_SLOTS_MODE ) ;

	Data_TxRF[1] = ISO15693_INVENTORY_RF_CMD ;		//Inventory Command
	if ( AFI_Mode == NON_CHECK_AFI_MODE )
	{
		Data_TxRF[2] = Mask_Len ;	//Mask Length
	
		Mask_Value_Pos = 3 ;
		LenData_TxRF =  3 ;
	}
	else
	{
		Data_TxRF[2] = AFI ; 		//AFI
		Data_TxRF[3] = Mask_Len ;	//Mask Length
	
		Mask_Value_Pos = 4 ;
		LenData_TxRF = 4 ;
	}

	Mask_Byte = Mask_Len / 8 ;
	if ( (Mask_Len % 8) != 0 )
	{	
		Mask_Byte = Mask_Byte + 1 ;	
	}

	for( i = 0 ; i < Mask_Byte ; i++ ) 
	{	
		Data_TxRF[Mask_Value_Pos + i] = *( Mask_Value + i ) ;		//Mask Length	
	}

	LenData_TxRF += Mask_Byte ;
	
	for( SlotMarker = 0 ; SlotMarker < 16 ; SlotMarker++ )
	{	
		// Slot Number
		if ( SlotMarker == 0x00 )
		{	
			Resp = BC45_Transceive( &Data_TxRF[0], LenData_TxRF, &Data_RxRF[0], &LenData_RxRF ) ;	
		}
		else	
		{	// Sending EOF only
			
			// Tx Config, Set bit Send 1 pulse, Bit 7
			BC45_Set_Send1Pulse_Bit() ;
			
			BC45_Flush_FIFO() ;

			if ( (Speed_Rx == RX_1_SUB_HIGH) || (Speed_Rx == RX_2_SUB_HIGH ) ) 
			{	
				Delay_ms(5) ; // (512*13*8 + 4192)/fc = 4.24ms
			}
			else
			{	//Low Data Rate
				Delay_ms(20) ; // (2048*13*8 + 4192)/fc = 16ms
			}

			Resp = BC45_Transceive( &Data_TxRF[0], 0, &Data_RxRF[0], &LenData_RxRF ) ;	
		}
	    
		if ( Resp == ASIC_EXE_TIMEOUT )
		{	// exit immediately
			BC45_Clear_Send1Pulse_Bit() ;			
			*(LenData_Resp) = 0 ;	
			
			return ASIC_EXE_TIMEOUT ;
		}
		else if ( Resp != NO_RESPONSE )
		{
			if ( Resp == _SUCCESS_ )
			{	// Check Len Resp
				if ( LenData_RxRF == LEN_ISO15693_INVENTORY_RF_RESP ) 
				{
					// Header
					*( Data_Resp + temp_LenData_Resp ) = SlotMarker ;  
			
					// FBP
					*( Data_Resp + temp_LenData_Resp + 1 ) = Resp ;  
			
					// Len ID
					*( Data_Resp + temp_LenData_Resp + 2 ) = (uint8_t) LenData_RxRF - 1 ;  
			
					// ID
					for( i = 0 ; i < LenData_RxRF - 1 ; i++ )
					{
						*( Data_Resp + temp_LenData_Resp + 3 + i )	= Data_RxRF[i + 1] ; 
					}
							
					temp_LenData_Resp += ( LenData_RxRF - 1 ) + 3 ;  
				}
				else
				{
					// Header
					*( Data_Resp + temp_LenData_Resp ) = SlotMarker ;  
			
					// FBP
					*( Data_Resp + temp_LenData_Resp + 1 ) = INVALID_RESP ;  
			
					// Len ID
					*( Data_Resp + temp_LenData_Resp + 2 ) = (uint8_t) LenData_RxRF ;  
			
					// ID
					for( i = 0 ; i < LenData_RxRF ; i++ )
					{
						*( Data_Resp + temp_LenData_Resp + 3 + i )	= Data_RxRF[i] ; 
					}
							
					temp_LenData_Resp += ( LenData_RxRF ) + 3 ;  				
				}
			}
			else // Fail
			{
				// Header
				*( Data_Resp + temp_LenData_Resp ) = SlotMarker ;  
		
				// FBP
				*( Data_Resp + temp_LenData_Resp + 1 ) = Resp ;  
		
				// Len ID
				*( Data_Resp + temp_LenData_Resp + 2 ) = (uint8_t) LenData_RxRF ;  
		
				// ID
				for( i = 0 ; i < LenData_RxRF ; i++ )
				{
					*( Data_Resp + temp_LenData_Resp + 3 + i )	= Data_RxRF[i] ; 
				}
						
				temp_LenData_Resp += ( LenData_RxRF ) + 3 ;  				
			}
		}
	};

	// Tx Config, Clear bit Send 1 pulse, Bit 7
	BC45_Clear_Send1Pulse_Bit() ;
	
	*(LenData_Resp) = temp_LenData_Resp ;	

	if ( temp_LenData_Resp == 0 )
	{
		return NO_RESPONSE ;
	}
	else
	{
		return _SUCCESS_ ;
	}
}

/*******************************************************************************
* Function Name	: ISO15693_Stay_Quiet
* Description 	: Perform Stay Quiet command in ISO15693 standard
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- No Response data  
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Stay_Quiet( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint16_t  i ;
	uint8_t   Resp ;
	
	// Request Flag
	Data_TxRF[0] = Request_Flag_ISO15693( Speed, Non_Inv_Mode, NON_INVENTORY_MODE ) ;
	Data_TxRF[1] = ISO15693_STAY_QUIET_RF_CMD ;		//Stay Quiet Command

	for ( i = 0 ; i < ISO15693_LEN_UID_BYTE ; i++ )
	{	
		Data_TxRF[i + 2] = *( UID + i ) ;		// UID 8 Bytes	
	}

	BC45_CRC_Setting( TxCRC_Enable, RxCRC_Enable ) ;

	Resp = BC45_Transceive( &Data_TxRF[0], 10, &Data_RxRF[0], &LenData_RxRF ) ;	

	if ( Resp == NO_RESPONSE )
	{
		Resp = _SUCCESS_ ;		
		*( LenData_Resp ) = 0 ;
	}
	else
	{
		Resp = INVALID_RESP ;		
		*( LenData_Resp ) = LenData_RxRF ;
		for ( i = 0; i < LenData_RxRF ; i++ )
		{
			*( Data_Resp + i ) = Data_RxRF[i] ;
		}
	}

	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Send_Data__NO_IC_MFG
* Description 	: Base function for Optional commands in ISO15693 Standard
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - UID_MODE
*						- UID_NOT_OPTIONAL(0x00)
*						- UID_OPTIONAL(0x01)
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Cmd_ISO15693 : Optional Command in ISO15693 Standard
*						- ISO15693_READ_SINGLE_BLOCK_RF_CMD(0x20)
*						- ISO15693_WRITE_SINGLE_BLOCK_RF_CMD(0x21)
*						- ISO15693_LOCK_BLOCK_RF_CMD(0x22)
*						- ISO15693_READ_MULTIPLE_BLOCK_RF_CMD(0x23)
*						- ISO15693_WRITE_MULTIPLE_BLOCK_RF_CMD(0x24)
*						- ISO15693_SELECT_RF_CMD(0x25)
*						- ISO15693_RESET_TO_READY_RF_CMD(0x26)
*						- ISO15693_WRITE_AFI_RF_CMD(0x27)
*						- ISO15693_LOCK_AFI_RF_CMD(0x28)
*						- ISO15693_WRITE_DSFID_RF_CMD(0x29)
*						- ISO15693_LOCK_DSFID_RF_CMD(0x2A)
*						- ISO15693_GET_SYSTEM_INFO_RF_CMD(0x2B)
*						- ISO15693_GET_MULTI_BLOCK_SECURE_RF_CMD(0x2C)
*				  - Param_ISO15693 : Parameter for each command
*				  - LenParam_ISO15693 : Length of Parameter for each command
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_ 
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Send_Data__NO_IC_MFG( uint8_t UID_MODE, uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t Cmd_ISO15693, uint8_t *Param_ISO15693, uint16_t LenParam_ISO15693, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint16_t  i ;
	uint8_t   Resp ;
	uint8_t   Operation_Mode ;

	Operation_Mode = Non_Inv_Mode & OPERATION_MODE_MASK ;	
	
	// Request Flag
	Data_TxRF[0] = Request_Flag_ISO15693( Speed, Non_Inv_Mode, NON_INVENTORY_MODE ) ;
	Data_TxRF[1] = Cmd_ISO15693 ;	//Command

	if ( (Operation_Mode == ADDR_MODE) || (UID_MODE == UID_NOT_OPTIONAL) )
	{
		for ( i = 0 ; i < ISO15693_LEN_UID_BYTE ; i++ )
		{	
			Data_TxRF[i + 2] = *( UID + i ) ;		// UID 8 Bytes	
		}
		LenData_TxRF = 10 ;
	}
	else
	{
		LenData_TxRF = 2 ;
	}
	
	for ( i = 0 ; i < LenParam_ISO15693 ; i++ )
	{
		Data_TxRF[LenData_TxRF++] = *( Param_ISO15693 + i ) ;	
	}
	
	BC45_CRC_Setting( TxCRC_Enable, RxCRC_Enable ) ;

	Resp = BC45_Transceive( &Data_TxRF[0], LenData_TxRF, &Data_RxRF[0], &LenData_RxRF ) ;	

	// --------------------------- Check Response ---------------------------------------------- //
	Resp = ISO15693_Check_Response( Resp, Data_RxRF, LenData_RxRF, Data_Resp, LenData_Resp ) ; 

	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Select
* Description 	: Perform Select command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from select command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Select( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param[1] ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_SELECT_RF_CMD, temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Reset_to_Ready
* Description 	: Perform Reset to ready command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from reset to ready command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- NO_RESPONSE
*						- FLAG_ERR 
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Reset_to_Ready( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param[1] ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_RESET_TO_READY_RF_CMD, temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Read_Single_Block
* Description 	: Perform Read single block command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Block_Num : Block Number
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Read single block command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- NO_RESPONSE
*						- FLAG_ERR 
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Read_Single_Block( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t Block_Num, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_READ_SINGLE_BLOCK_RF_CMD, &Block_Num, 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Write_Single_Block
* Description 	: Perform Write single block command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Block_Size : the number of byte data in each block ( which may be different in tag's brand ) 
*				  - Write_Block_Param : 
*						- 1st byte : Block Number 
*						- 2nd byte to Late byte : Data to be written 
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from write single block command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Write_Single_Block( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t Block_Size, uint8_t *Write_Block_Param, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	// Write_Block_Param = Block_Num ( 1 Bytes ) + Data_Wr ( N Bytes )
	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_WRITE_SINGLE_BLOCK_RF_CMD, Write_Block_Param, Block_Size + 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Lock_Block
* Description 	: Perform Lock block command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Block_Num : Block Number to be locked
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from lock block command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Lock_Block( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t Block_Num, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_LOCK_BLOCK_RF_CMD, &Block_Num, 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Read_Multiple_Blocks
* Description 	: Perform Read multiple blocks command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Read_Multi_Block_Param : 
*						- 1st byte : Block Number to be read
*						- 2nd byte : The Number of block to be read
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from read multiple blocks command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Read_Multiple_Blocks( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Read_Multi_Block_Param, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	// Read_Block_Param = Block_Num ( 1 Bytes ) + Num_Block ( 1 Bytes )  =  2 Bytes
	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_READ_MULTIPLE_BLOCK_RF_CMD, Read_Multi_Block_Param, 2, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Write_Multiple_Blocks
* Description 	: Perform Write multiple blocks command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Block_Size 
*				  - Write_Multi_Block_Param : 
*						- 1st byte : First Block Number to be written
*						- 2nd byte : The Number of block to be written 
*						- 3rd to Last byte : Data to be written( N bytes )
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from write multiple blocks command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Write_Multiple_Blocks( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t Block_Size, uint8_t *Write_Multi_Block_Param, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Num_Block ;
	uint16_t  temp_LenParam_Send ;
	// Write_Block_Param = First_Block_Num ( 1 Bytes ) + Num_Block ( 1 Bytes ) + Data_Wr ( N Bytes )
	temp_Num_Block = *( Write_Multi_Block_Param + 1 ) ; 
	temp_LenParam_Send = ( ( temp_Num_Block + 1 ) * Block_Size ) + 2 ; // Add 2 ==> First_Block_Num ( 1 Bytes ) + Num_Block ( 1 Bytes ) = 2 Bytes
	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_WRITE_MULTIPLE_BLOCK_RF_CMD, Write_Multi_Block_Param, temp_LenParam_Send, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Write_AFI
* Description 	: Perform Write AFI command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - AFI_Value 
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from write AFI command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Write_AFI( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t AFI_Value, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_WRITE_AFI_RF_CMD, &AFI_Value, 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Lock_AFI
* Description 	: Perform Lock AFI command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from lock AFI command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Lock_AFI( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	
	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_LOCK_AFI_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Write_DSFID
* Description 	: Perform Write DSFID command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - DSFID_Value 
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from write DSFID command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Write_DSFID( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t DSFID_Value, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_WRITE_DSFID_RF_CMD, &DSFID_Value, 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Lock_DSFID
* Description 	: Perform Lock DSFID command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from lock DSFID command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Lock_DSFID( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_LOCK_DSFID_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Get_System_Information
* Description 	: Perform Get_system_information command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from get system information command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Get_System_Information( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_GET_SYSTEM_INFO_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Get_Multiple_Block_Security_Status
* Description 	: Perform Get_multiple_block_security_status command in ISO15693
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inv_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						Bit 7 : RFU_Flag(Bit in request flag of ISO15693)
*						Bit 6 : Option_Flag(Bit in request flag of ISO15693)
*						Bit 5 : Protocol_Extension_Flag(Bit in request flag of ISO15693)
*						Bit 5 : RFU
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Get_Multi_Block_Secure_Param : 
*						- 1st byte : Block Number
*						- 2nd byte : Number of block
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Get multiple block security status command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Get_Multiple_Block_Security_Status( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t *UID, uint8_t *Get_Multi_Block_Secure_Param, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	// Get_Multi_Block_Secure_Param = First_Block_Num ( 1 Bytes ) + Num_Block ( 1 Bytes )  =  2 Bytes
	Resp = ISO15693_Send_Data__NO_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, UID, ISO15693_GET_MULTI_BLOCK_SECURE_RF_CMD, Get_Multi_Block_Secure_Param, 2, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//



//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//
//--------------------------HTK Custom Cmd ---------------------------------------------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------//

/*******************************************************************************
* Function Name	: ISO15693_Send_Data__WITH_IC_MFG
* Description 	: Base function for Custom commands in ISO15693 Standard
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - UID_MODE
*						- UID_NOT_OPTIONAL(0x00) : UID is mendatory if UID_MODE = UID_NOT_OPTIONAL
*						- UID_OPTIONAL(0x01)
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Silicon Craft
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Cmd_ISO15693 : Command in ISO15693 Standard
*						- ISO15693_SET_EAS_HTK_RF_CMD(0xA2)
*						- ISO15693_RESET_EAS_HTK_RF_CMD(0xA3)
*						- ISO15693_LOCK_EAS_HTK_RF_CMD(0xA4)
*						- ISO15693_EAS_ALARM_HTK_RF_CMD(0xA5)
*						- ISO15693_KILL_HTK_RF_CMD(0xA6)
*						- ISO15693_SET_PASSWORD_HTK_RF_CMD(0xB3)
*						- ISO15693_WRITE_PASSWORD_HTK_RF_CMD(0xB4)
*						- ISO15693_LOCK_PASSWORD_HTK_RF_CMD(0xB5)
*						- ISO15693_SET_PASSWORD_MODE_HTK_RF_CMD(0xB6)
*						- ISO15693_LOCK_PASSWORD_MODE_HTK_RF_CMD(0xB7)
*						- ISO15693_GET_PASSWORD_MODE_HTK_RF_CMD(0xB8)
*						- ISO15693_SET_OTP_HTK_RF_CMD(0xC0)
*						- ISO15693_GET_OTP_HTK_RF_CMD(0xC1)
*						- ISO15693_WRITE_OTP_UID_HTK_RF_CMD(0xC2)
*						- ISO15693_READ_OTP_UID_HTK_RF_CMD(0xC3)
*				  - Param_ISO15693 : Parameter for each command
*				  - LenParam_ISO15693 : Length of Parameter for each command
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_ 
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Send_Data__WITH_IC_MFG( uint8_t UID_MODE, uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t Cmd_ISO15693, uint8_t *Param_ISO15693, uint16_t LenParam_ISO15693, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint16_t  i ;
	uint8_t   Resp ;
	uint8_t   Operation_Mode ;

	Operation_Mode = Non_Inv_Mode & OPERATION_MODE_MASK ;	
	
	// Request Flag
	Data_TxRF[0] = Request_Flag_ISO15693( Speed, Non_Inv_Mode, NON_INVENTORY_MODE ) ;
	Data_TxRF[1] = Cmd_ISO15693 ;	//Command
	Data_TxRF[2] = IC_MFG ;	

	if ( (Operation_Mode == ADDR_MODE) || (UID_MODE == UID_NOT_OPTIONAL) )
	{
		for ( i = 0 ; i < ISO15693_LEN_UID_BYTE ; i++ )
		{	
			Data_TxRF[i + 3] = *( UID + i ) ;		// UID 8 Bytes	
		}
		LenData_TxRF = 11 ;
	}
	else
	{
		LenData_TxRF = 3 ;
	}
	
	for ( i = 0 ; i < LenParam_ISO15693 ; i++ )
	{
		Data_TxRF[LenData_TxRF++] = *( Param_ISO15693 + i ) ;	
	}
	
	BC45_CRC_Setting( TxCRC_Enable, RxCRC_Enable ) ;

	Resp = BC45_Transceive( &Data_TxRF[0], LenData_TxRF, &Data_RxRF[0], &LenData_RxRF ) ;	

	// --------------------------- Check Response ---------------------------------------------- //
	Resp = ISO15693_Check_Response( Resp, Data_RxRF, LenData_RxRF, Data_Resp, LenData_Resp ) ; 

	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Set_EAS_HTK
* Description 	: Perform Set EAS HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Set EAS HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Set_EAS_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_SET_EAS_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Reset_EAS_HTK
* Description 	: Perform Reset EAS HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Silicon Craft
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from EAS HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Reset_EAS_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_RESET_EAS_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Lock_EAS_HTK
* Description 	: Perform Lock EAS HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Lock EAS HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Lock_EAS_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_LOCK_EAS_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_EAS_Alarm_HTK
* Description 	: Perform EAS Alarm command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from EAS Alarm command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_EAS_Alarm_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_EAS_ALARM_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Kill_HTK
* Description 	: Perform Kill HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Kill_Code : Kill code 4 bytes 
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Kill HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Kill_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Kill_Code, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_KILL_HTK_RF_CMD, Kill_Code, 4, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_CalCRC
* Description 	: Calculate CRC for Set_password Command of SIC5600 
* 				  This function is Base function for  GEN_RNG_PWD which is only used for SIC5600 
* Input         : 
*				  - CRC_Preset_MSB
*				  - CRC_Preset_LSB
*				  - Data
*				  - LenData
* Output		: 
*				  - CRC1 : CRC16 High byte
*				  - CRC0 : CRC16 Low byte
* Return        : None
*******************************************************************************/
void ISO15693_CalCRC( uint8_t CRC_Preset_MSB, uint8_t CRC_Preset_LSB, uint8_t *Data, uint16_t LenData, uint8_t *CRC1, uint8_t *CRC0 )
{	
	uint8_t	  i ;
	uint16_t  wCRC ;

	wCRC =  ( ( (uint16_t)CRC_Preset_MSB ) << 8 ) | ( (uint16_t)CRC_Preset_LSB ) ;
	wCRC = Reverse_Data_16Bit ( wCRC ) ;
	
	for ( i = 0 ; i < LenData ; i++ )
	{	
		UpdateCRC( *(Data + i), &wCRC ) ;
	};

	wCRC = Reverse_Data_16Bit(wCRC) ;
	*( CRC1 ) = ( uint8_t )( (wCRC >> 8) & 0xFF ) ;
	*( CRC0 ) = ( uint8_t )( wCRC & 0xFF ) ;
}

/*******************************************************************************
* Function Name	: Gen_RNG_PWD
* Description 	: Generate RNG_PWD which is a required argument in Set_Password command in SIC5600 
* Input         : 
*				  - Pwd_Mode : "00" + Kill(1 bit) + M(1 bit) + PA(2 bits) + SM(2 bits) 
*						Kill : (Bit5) Don't care in this function
*							- 0b : Kill Enable
*							- 1b : Kill Disable
*						M : Parameter indicating the current protection password in SIC5600 (Bit4)
*							this parameter is only required during accessing SIC5600's content.
*							- 0b : Use address PWD0
*							- 1b : Use address PWD1
*						PA : Password Allocation (Bit3:2)   
*							- 00b : Use Password 0 for Protected Read and Password 1 for Protected Write  (M = 0,  PWD 0) (M = 1,  PWD 1) 
*							- 01b : Use Password 0   
*							- 10b : Use Password 1   
*							- 11b : Use Both Password 0 and Password 1 (64 Bits) (PWD0+PWD1) 
*						SM : Security Mode (Bit1:0)    
*							- 00b : Normal   
*							- 01b : Protected Read   
*							- 10b : Protected Write   
*							- 11b : Protected Read and Write   
*				  - RNG : Random number generation 4 or 8 bytes depending on Pwd_Mode
*				  - Pwd : Password 4 or 8 bytes depend on Pwd_Mode
* Output		: 
*				  - RNG_PWD : Packed data for Set_Password command in SIC5600
*				  - Len_RNG_PWD : Length of RNG_PWD 
* Return        : None
*******************************************************************************/
void Gen_RNG_PWD( uint8_t Pwd_Mode, uint8_t *RNG, uint8_t *Pwd, uint8_t *RNG_PWD, uint8_t *Len_RNG_PWD ) 
{
	uint8_t  i ;
	uint8_t  PA ;
	uint8_t  PWD_CRC[2] ;
	uint8_t  Loop_Cal ;
	
	// Pwd_Mode = M + PA + SM
	
	PA =  ( Pwd_Mode & PA_MASK ) >> 2 ;
	if ( PA == PM_USE_PWD_64_PROTECT_ALL ) 
	{	
		Loop_Cal = 4 ;	
		*( Len_RNG_PWD ) = 16 ;
	}
	else
	{
		Loop_Cal = 2 ;	
		*( Len_RNG_PWD ) = 8 ;
	}
	
	for ( i = 0 ; i < Loop_Cal ; i++ )
	{
		ISO15693_CalCRC( *( RNG + (2 * i) + 1 ), *( RNG + (2 * i) ), ( Pwd + (2 * i) ), 2, ( PWD_CRC + 1 ), ( PWD_CRC ) ) ;
		*( RNG_PWD + (i * 4) ) 	   = *( RNG + (2 * i) ) ;
		*( RNG_PWD + (i * 4) + 1 ) = *( RNG + (2 * i) + 1 ) ;
		*( RNG_PWD + (i * 4) + 2 ) = *( PWD_CRC ) ;
		*( RNG_PWD + (i * 4) + 3 ) = *( PWD_CRC + 1 ) ;
	}
}

/*******************************************************************************
* Function Name	: ISO15693_Set_Password_HTK
* Description 	: Perform Set password HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Silicon Craft
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Pwd_Mode : "00" + Kill(1 bit) + M(1 bit) + PA(2 bits) + SM(2 bits) 
*						Kill : (Bit5) Don't care in this function
*							- 0b : Kill Enable
*							- 1b : Kill Disable
*						M : Parameter indicating the current protection password in SIC5600 (Bit4)
*							- 0b : Use address PWD0
*							- 1b : Use address PWD1
*						PA : Password Allocation (Bit3:2)   
*							- 00b : Use Password 0 for Protected Read and Password 1 for Protected Write  (M = 0,  PWD 0) (M = 1,  PWD 1) 
*							- 01b : Use Password 0   
*							- 10b : Use Password 1   	
*							- 11b : Use Both Password 0 and Password 1 (64 Bits) = (PWD0+PWD1) 
*						SM : Security Mode (Bit1:0)    
*							- 00b : Normal   
*							- 01b : Protected Read   
*							- 10b : Protected Write   		
*							- 11b : Protected Read and Write   
*				  - RNG : Random number generation 4 or 8 bytes depend on Pwd_Mode
*				  - Pwd : Password 4 or 8 bytes depend on Pwd_Mode
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Set password HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Set_Password_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t Pwd_Mode, uint8_t *RNG, uint8_t *Pwd, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  i ;
	uint8_t   Resp ;
	uint8_t   Len_RNG_PWD ;
	uint8_t   RNG_PWD[16] ;
	uint8_t   temp_Param[20] ;
	
	// Cal RNG_Pwd
	Gen_RNG_PWD( Pwd_Mode, RNG, Pwd, RNG_PWD, &Len_RNG_PWD ) ;	

	temp_Param[0] = Pwd_Mode ;			
	for ( i = 0 ; i < Len_RNG_PWD ; i++ )
	{	
		temp_Param[i + 1] = *( RNG_PWD + i ) ;		// RNG_PWD N Bytes	
	}

	// Set_Password_Param = Pwd_ID ( 1 Byte ) + Pwd ( 4 Bytes ) = 5 Bytes
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_SET_PASSWORD_HTK_RF_CMD, temp_Param, (Len_RNG_PWD + 1), Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Write_Password_HTK
* Description 	: Perform Write password HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Write_Password_Param :
*						- 1st byte : Pwd_ID
*										- 0x01h = Password 0 Addr
*										- 0x02h = Password 1 Addr
*										- 0x03h = Kill Code Addr
*						- 2nd to Last byte : Password (4 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Write password HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Write_Password_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Write_Password_Param, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	
	// Write_Password_Param = Pwd_ID ( 1 Byte ) + Pwd ( 4 Bytes ) = 5 Bytes
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_WRITE_PASSWORD_HTK_RF_CMD, Write_Password_Param, 5, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Lock_Password_HTK
* Description 	: Perform Lock password HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Pwd_ID
*						- 0x01h = Password 0 Addr
*						- 0x02h = Password 1 Addr
*						- 0x03h = Kill Code Addr
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Lock password HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Lock_Password_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t Pwd_ID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	
	// Lock_Password_Param = Pwd_ID ( 1 Byte ) = 1 Byte
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_LOCK_PASSWORD_HTK_RF_CMD, &Pwd_ID, 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Set_Password_Mode_HTK
* Description 	: Perform Set password HTK command (SIC5600)
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - Pwd_Mode : "00" + Kill(1 bit) + M(1 bit) + PA(2 bits) + SM(2 bits) 
*						Kill : (Bit5)
*							- 0b : Kill Enable
*							- 1b : Kill Disable
*						M :	Parameter indicating the current protection password in SIC5600 (Bit4)
*							this parameter is only required during accessing SIC5600's content.
*							- 0b : Use address PWD0
*							- 1b : Use address PWD1
*						PA : Password Allocation (Bit3:2)   
*							- 00b : Use Password 0 for Protected Read and Password 1 for Protected Write  (M = 0,  PWD 0) (M = 1,  PWD 1) 
*							- 01b : Use Password 0   
*							- 10b : Use Password 1   	
*							- 11b : Use Both Password 0 and Password 1 (64 Bits) = ( PWD0+PWD1) 
*						SM : Security Mode (Bit1:0)    
*							- 00b : Normal   
*							- 01b : Protected Read   
*							- 10b : Protected Write   		
*							- 11b : Protected Read and Write   
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Set password mode HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Set_Password_Mode_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t Pwd_Mode, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	
	// Set_Password_Mode_Param = Pwd_Mode ( 1 Byte ) = 1 Byte
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_SET_PASSWORD_MODE_HTK_RF_CMD, &Pwd_Mode, 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Lock_Password_Mode_HTK
* Description 	: Perform Lock password mode HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Lock password mode HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Lock_Password_Mode_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_LOCK_PASSWORD_MODE_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Get_Password_Mode_HTK
* Description 	: Perform Get password mode HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Get password mode HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Get_Password_Mode_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;

	
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_GET_PASSWORD_MODE_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Set_OTP_HTK
* Description 	: Perform Set OTP HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - OTP_Mode : '00000' + L(1 bit) + MODE(2 bits)
*						L : Lock OTP (Bit2)
*							- 0b = Not Lock OTP
*							- 1b = Lock OTP
*						MODE : OTP Mode Lock (Bit1:0)
*							- 00b = Not Lock
*							- 01b = Lock Page 0 (Address 0 - 15)
*							- 10b = Lock Page 1 (Address 16 - 31)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Set OTP HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Set_OTP_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t OTP_Mode, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	
	// Param = OTP_Mode ( 1 Byte ) = 1 Byte
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_SET_OTP_HTK_RF_CMD, &OTP_Mode, 1, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Get_OTP_HTK
* Description 	: Perform Get OTP HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Get OTP HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Get_OTP_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;
	
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_GET_OTP_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Write_OTP_UID_HTK
* Description 	: Perform Write OTP UID HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
*				  - OTP_UID : OTP UID (6 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Write OTP UID HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Write_OTP_UID_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *OTP_UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	
	// Param = OTP_UID ( 6 Byte ) = 6 Byte
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_NOT_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_WRITE_OTP_UID_HTK_RF_CMD, OTP_UID, 6, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Read_OTP_UID_HTK
* Description 	: Perform Read OTP UID HTK command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : 
*				  - Speed : Tx_Speed(4 bits) + RxSpeed(4 bits) (This parameter is also the same value for function BC45_Speed_15693) 
*						High Nibble : Tx_Speed(Bit7:4)
*							- 0000b : 1 out of 256 (1.65 kbits/s)
*							- 0001b : 1 out of 4 (26.48 kbits/s) 
*							- Other : 1 out of 4 (26.48 kbits/s) 
*						Low Nibble : RxSpeed(Bit3:0)				
*							- 0000b : 1 sub low (6.62 kbits/s)   
*							- 0001b : 1 sub high (26.48 kbits/s)
*							- 0010b : 1 sub ultra high (52.96 kbits/s)
*							- 0011b : 2 sub low (6.67 kbits/s)
*							- 0100b : 2 sub high (26.69 kbits/s)
*							- Other : 1 sub high (26.69 kbits/s)
*				  - Non_Inventory_Mode : RFU_Flg(1 bit) + Option_Flg(1 bit) + Protocol_Extension_Flg(1 bit) + '0' + Operation mode(4 bits)
*						RFU_Flag : Bit in request flag of ISO15693 (Bit7)
*							- 0b = Clear RFU_Flag bit
*							- 1b = Set RFU_Flag bit
*						Option_Flag : Bit in request flag of ISO15693 (Bit6)
*							- 0b = Clear Option_Flag bit
*							- 1b = Set Option_Flag bit
*						Protocol_Extension_Flag : Bit in request flag of ISO15693 (Bit5)
*							- 0b = Clear Protocol_Extension_Flag bit
*							- 1b = Set Protocol_Extension_Flag bit
*						Low Nibble : Operation mode (Bit3:0)		 
*							- 0000b : Non Address mode
*							- 0001b : Address mode
*							- 0010b : Select mode               
*				  - IC_MFG : IC manufacturer code
*						- IC_MFG_HTK(0x39) : Holtek
*				  - UID : Unique identifier of ISO15693 (8 bytes)
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data from Read OTP UID HTK command
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Read_OTP_UID_HTK( uint8_t Speed, uint8_t Non_Inv_Mode, uint8_t IC_MFG, uint8_t *UID, uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint8_t  Resp ;
	uint8_t  temp_Param ;
	
	Resp = ISO15693_Send_Data__WITH_IC_MFG( UID_OPTIONAL, Speed, Non_Inv_Mode, IC_MFG, UID, ISO15693_READ_OTP_UID_HTK_RF_CMD, &temp_Param, 0, Data_Resp, LenData_Resp ) ;
	
	return Resp ;
}

/*******************************************************************************
* Function Name	: ISO15693_Send_1_Pulse
* Description 	: Perform Send 1 pulse command
*				: Require correct speed setting before runing this function by call function BC45_Speed_15693
* Input         : None
* Output		: 
*				  - Data_Resp : Response data
*						- if Response = _SUCCESS_
*							- 1st to Last byte : Response data 
*						- if Response = FLAG_ERR
*							- 1st : Error code following ISO15693 Standard
*						- if Response = FRAMING_ERR or COLLISION_ERR or PARITY_ERR or CRC_ERR or INVALID_RESP (RF_COMMU_ERR_CATEGORY)
*							- 1st to Last byte : Error data in BC45 FIFO (if it exists)
*						- Other (NO_RESPONSE, ASIC_EXE_TIMEOUT)
*							- No Response data  
*				  - LenData_Resp : Length of Response data
* Return        : 
*				  - Response
*						- _SUCCESS_  
*						- FLAG_ERR 
*						- NO_RESPONSE
*						- FRAMING_ERR
*						- COLLISION_ERR
*						- PARITY_ERR
*						- CRC_ERR
*						- INVALID_RESP
*						- ASIC_EXE_TIMEOUT  
*******************************************************************************/
uint8_t ISO15693_Send_1_Pulse( uint8_t *Data_Resp, uint16_t *LenData_Resp )
{
	uint16_t  i ;
	uint8_t   Resp ;

	BC45_Set_Send1Pulse_Bit() ;
	BC45_CRC_Setting( TxCRC_Enable, RxCRC_Enable ) ;
	Resp = BC45_Transceive( &Data_TxRF[0], 0, &Data_RxRF[0], &LenData_RxRF ) ;	
	BC45_Clear_Send1Pulse_Bit() ;
	
	*(LenData_Resp) = LenData_RxRF ;	
	for( i = 0 ; i < LenData_RxRF ; i++ )
	{
		*( Data_Resp + i ) = Data_RxRF[i] ;		
	}

	return Resp ;
}
